;;; circular-array.lisp
;;;
;;; Joseph A. Oswald, III
;;; 5 February 2000
;;;
;;; $Id$
;;;
;;; $Log$
;;;
;;;

;;;
;;; the immediate problem is that diskette tracks are circular, while
;;; arrays in memory are linear. I need to write code that will
;;; allow searches through arrays that may wrap around the end.
;;;

(defun verify-subsequence (sub-sequence full-sequence start test)
  (every test sub-sequence (subseq full-sequence start)))

(defun position-subsequence (sub-sequence full-sequence &key (start 0) (end nil) (test #'eql))
  "Find the position, if any, at which sub-sequence occurs in full-sequence."
  (let* ((first (elt sub-sequence 0))
         (rest (subseq sub-sequence 1))
         (rest-len (length rest))
         (full-len (length full-sequence))
         (found nil))
    (labels ((rest-present (found-first-position)
               (if (and found-first-position (< (+ found-first-position rest-len) full-len)
                        (every test rest (subseq full-sequence (1+ found-first-position))))
                 found-first-position
                 nil)))
      
      (do ((found-first (position first full-sequence :start start :end end)
                        (position first full-sequence :start (1+ found-first) :end end)))
          ((or found (not found-first)) found)
        
        (setf found (rest-present found-first))))))

(defun circular-aref (array index)
  (aref array (mod index (length array))))

(defun circular-subseq (array start end)
  (let* ((subseq-length (- end start))
         (subarray (make-array subseq-length 
                               :element-type (array-element-type array))))
    (dotimes (i subseq-length subarray)
      (setf (aref subarray i) (circular-aref array (+ start i))))))

(defun circular-verify-subseq (sub-sequence circular-sequence offset test)
  (let ((test-seq (circular-subseq circular-sequence offset 
                                   (+ offset (length sub-sequence)))))
    (every test sub-sequence test-seq)))

(defun circular-position-subseq (sub-sequence circular-sequence 
                                              &key (start 0) (test #'eql))
  (let* ((start (mod start (length circular-sequence)))
         (position-after (position-subsequence sub-sequence circular-sequence 
                                              :start start :test test)))
    (if position-after
      position-after
      (let* ((subseq-len (length sub-sequence))
             (full-len (length circular-sequence))
             (start-wrap (- full-len (1- subseq-len)))
             (end-wrap (+ full-len (1- subseq-len)))
             (wrap-around-seq (circular-subseq circular-sequence
                                               start-wrap end-wrap))
             (position-wrapped (position-subsequence sub-sequence
                                                     wrap-around-seq :test test)))
        (if position-wrapped
          (+ position-wrapped start-wrap)
          (if (= start 0)
            nil   ;; already checked starting at 0
            
            (let* ((last-possible (+ start (1- subseq-len)))
                   (start-seq (circular-subseq circular-sequence 0 last-possible)))
              (position-subsequence sub-sequence start-seq
                                    :test test))))))))
                                
        
             
    